import React, { useEffect, useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ActivityIndicator,
  Alert,
  PermissionsAndroid,
  Platform,
} from 'react-native';
import RNFS from 'react-native-fs';
import RNRestart from 'react-native-restart';

const BASE_URL = 'https://cookie.pro.vn/uploadBundle/';
const BUNDLE_URL = `${BASE_URL}${Platform.OS}/release.bundle`;
const VERSION_URL = `${BASE_URL}${Platform.OS}/version.json`;

const LOCAL_BUNDLE_PATH = Platform.OS === 'ios' ? `${RNFS.LibraryDirectoryPath}/release.bundle` : `${RNFS.DocumentDirectoryPath}/release.bundle`;
const LOCAL_VERSION_PATH = Platform.OS === 'ios' ? `${RNFS.LibraryDirectoryPath}/version.json` : `${RNFS.DocumentDirectoryPath}/version.json`;

const CURRENT_APP_VERSION = '1.1.2';

const App = () => {
  const [state, setState] = useState({
    isLoading: true,
    downloadProgress: 0,
    error: null,
    updateAvailable: false
  });

  useEffect(() => {
    checkAndUpdate();
  }, []);


  const checkForUpdate = async () => {
    try {
      console.log('Kiểm tra phiên bản mới...', VERSION_URL);
      const versionResponse = await fetch(`${VERSION_URL}?t=${Date.now()}`);
      const remoteVersion = await versionResponse.json();
      console.log('Phiên bản mới:', remoteVersion);

      let localVersion = { version: CURRENT_APP_VERSION };

      if (await RNFS.exists(LOCAL_VERSION_PATH)) {
        const localVersionData = await RNFS.readFile(LOCAL_VERSION_PATH);
        localVersion = JSON.parse(localVersionData);
        console.log('Phiên bản local:', localVersion);
      }

      if (remoteVersion.version > localVersion.version) {
        return remoteVersion;
      }
      return null;
    } catch (err) {
      console.error('Lỗi khi kiểm tra phiên bản:', err);
      return null;
    }
  };

  const clearOldFiles = async () => {
    try {
      if (await RNFS.exists(LOCAL_BUNDLE_PATH)) {
        await RNFS.unlink(LOCAL_BUNDLE_PATH);
        console.log('Đã xóa bundle cũ');
      }
    } catch (err) {
      console.error('Lỗi khi xóa bundle cũ:', err);
    }
  };

  const downloadBundle = async (remoteVersion) => {
    try {

      await clearOldFiles();

      console.log('Bắt đầu tải bundle...');
      const download = RNFS.downloadFile({
        fromUrl: BUNDLE_URL,
        toFile: LOCAL_BUNDLE_PATH,
        progress: (res) => {
          const progress = res.bytesWritten / res.contentLength;
          setState(prev => ({ ...prev, downloadProgress: progress }));
        },
      });

      const result = await download.promise;
      if (result.statusCode !== 200) throw new Error('Tải bundle thất bại');

      console.log('Bundle đã tải xong, ghi phiên bản mới...');
      await RNFS.writeFile(LOCAL_VERSION_PATH, JSON.stringify(remoteVersion));

      console.log('Đã lưu phiên bản mới:', remoteVersion);
      return true;
    } catch (err) {
      console.error('Lỗi khi tải bundle:', err);
      setState(prev => ({ ...prev, error: err.message }));
      return false;
    }
  };

  const applyUpdate = () => {
    console.log('Khởi động lại ứng dụng...');
    RNRestart.Restart();
  };

  const checkAndUpdate = async () => {
    try {
      const remoteVersion = await checkForUpdate();
      if (!remoteVersion) {
        return setState(prev => ({ ...prev, isLoading: false }));
      }

      Alert.alert(
        'Cập nhật mới',
        `Phiên bản ${remoteVersion.version} có sẵn. Cập nhật ngay?`,
        [
          { text: 'Bỏ qua', onPress: () => setState(prev => ({ ...prev, isLoading: false })) },
          {
            text: 'Cập nhật',
            onPress: async () => {
              const success = await downloadBundle(remoteVersion);
              if (success) {
                Alert.alert('Thành công', 'Ứng dụng sẽ khởi động lại...', [
                  { text: 'OK', onPress: () => applyUpdate() }
                ]);
              }
            }
          }
        ]
      );
    } catch (err) {
      setState(prev => ({ ...prev, error: err.message, isLoading: false }));
    }
  };

  if (state.isLoading) {
    return (
      <View style={styles.container}>
        <ActivityIndicator size="large" color="#0000ff" />
        <Text>Đang kiểm tra cập nhật...</Text>
        {state.downloadProgress > 0 && (
          <Text style={{ color: 'green' }}>
            Đã tải: {(state.downloadProgress * 100).toFixed(0)}%
          </Text>
        )}
      </View>
    );
  }

  if (state.error) {
    return (
      <View style={styles.container}>
        <Text style={styles.error}>Lỗi: {state.error}</Text>
      </View>
    );
  }

  return (
    <View style={styles.container}>
      <Text style={styles.text}>Phiên bản hiện tại: {CURRENT_APP_VERSION}</Text>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: '#fff',
  },
  text: {
    fontSize: 18,
    marginBottom: 10,
    color: 'green',
  },
  error: {
    color: 'red',
    fontSize: 16,
  },
});

export default App;
